<?php
// File: /admin/deposits.php
require_once '../includes/db.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';

requireAdmin();

// Handle Actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $deposit_id = $_POST['deposit_id'];
    $action = $_POST['action'];
    
    $stmt = $pdo->prepare("SELECT * FROM deposits WHERE id = ?");
    $stmt->execute([$deposit_id]);
    $deposit = $stmt->fetch();

    if ($deposit && $deposit['status'] == 'pending') {
        if ($action === 'approve') {
            try {
                $pdo->beginTransaction();
                
                // Update Deposit Status
                $stmt = $pdo->prepare("UPDATE deposits SET status = 'approved' WHERE id = ?");
                $stmt->execute([$deposit_id]);
                
                // Add Balance to User
                $stmt = $pdo->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
                $stmt->execute([$deposit['amount'], $deposit['user_id']]);
                
                // Log Transaction
                logTransaction($pdo, $deposit['user_id'], 'deposit', $deposit['amount'], "Deposit Approved");
                
                $pdo->commit();
                setFlash('success', 'Deposit approved successfully.');
            } catch (Exception $e) {
                $pdo->rollBack();
                setFlash('danger', 'Error: ' . $e->getMessage());
            }
        } elseif ($action === 'reject') {
            $stmt = $pdo->prepare("UPDATE deposits SET status = 'rejected' WHERE id = ?");
            $stmt->execute([$deposit_id]);
            setFlash('info', 'Deposit rejected.');
        }
    }
    redirect('deposits.php');
}

// Fetch Deposits
$stmt = $pdo->query("SELECT d.*, u.username FROM deposits d JOIN users u ON d.user_id = u.id ORDER BY d.created_at DESC");
$deposits = $stmt->fetchAll();

$page_title = "Manage Deposits";
include '../includes/header.php';
?>

<div class="card">
    <h3 class="mb-4">Deposit Requests</h3>
    <div class="table-responsive">
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>User</th>
                    <th>Amount</th>
                    <th>Method</th>
                    <th>Status</th>
                    <th>Date</th>
                    <th>Action</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($deposits as $d): ?>
                <tr>
                    <td>#<?php echo $d['id']; ?></td>
                    <td><?php echo htmlspecialchars($d['username']); ?></td>
                    <td class="text-success"><?php echo formatMoney($d['amount']); ?></td>
                    <td><?php echo htmlspecialchars($d['method']); ?></td>
                    <td>
                        <span class="badge badge-<?php 
                            echo $d['status'] == 'approved' ? 'success' : 
                                ($d['status'] == 'rejected' ? 'danger' : 'warning'); 
                        ?>">
                            <?php echo ucfirst($d['status']); ?>
                        </span>
                    </td>
                    <td><?php echo date('M d, Y', strtotime($d['created_at'])); ?></td>
                    <td>
                        <?php if ($d['status'] == 'pending'): ?>
                        <form method="POST" style="display:inline-block;">
                            <input type="hidden" name="deposit_id" value="<?php echo $d['id']; ?>">
                            <button type="submit" name="action" value="approve" class="btn btn-success" style="padding: 5px 10px; font-size: 0.8rem;">Approve</button>
                            <button type="submit" name="action" value="reject" class="btn btn-danger" style="padding: 5px 10px; font-size: 0.8rem;">Reject</button>
                        </form>
                        <?php else: ?>
                            -
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
